package pcshoppingassistant;

import net.sourceforge.selectionengine.*;
//import temp.*;
import java.util.*;

public class DataBreakdownForTraits {
//-----------------
//--- Constants ---
//-----------------
    private final int stepIncrement = 10;   //--- 10 points - 10%, 20%, 30%, etc.
    private final int numberOfSteps = (100/stepIncrement)+1;    //--- 0-100
                                                                //--- +1 for 0%

//------------------
//--- Attributes ---
//------------------
    //--- HashMap = key can be trait name, no sorting
    //--- ArrayList = can sort, key must be int
    //--- Collection of DataBreakdownForTrait objects, one per trait
    private HashMap breakdowns = new HashMap(); //--- key=trait name
    private HashMap minValues = new HashMap();  //--- key=trait name
    private HashMap maxValues = new HashMap();  //--- key=trait name


//--------------------
//--- Constructors ---
//--------------------
    public DataBreakdownForTraits(Items items) {
        parseItems(items);
    }


//---------------
//--- Methods ---
//---------------
    protected void addDataToBreakPoints(Items items) {
        try {
            TraitDescriptors traitDescriptors = items.getTraitDescriptors();

            //--- For each Item in Items
            Iterator itemsIterator = items.iterator();
            while (itemsIterator.hasNext()) {
                Item item = (Item) itemsIterator.next();

                //--- For each Trait in Item
                Iterator traitsIterator = item.iterator();
                while (traitsIterator.hasNext()) {
                    Trait trait = (Trait) traitsIterator.next();
                    String traitName = trait.getName();
                    TraitValue value = trait.getValue();

                    DataBreakdownForTrait dataBreakdown =
                        (DataBreakdownForTrait) getDataBreakdown(traitName);
                    dataBreakdown.addDataPoint(value.toString());
                }  //--- traitIterator.hasNext()
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }  //--- addDataToBreakPoints


    protected void determineMinAndMax(Items items) {
        try {
            TraitDescriptors descriptors = items.getTraitDescriptors();

            //--- Go through each item
            Iterator itemsIterator = items.iterator();
            while (itemsIterator.hasNext()) {
                Item item = (Item) itemsIterator.next();

                //--- For each trait in this item
                //---   look for the max and min for each trait
                Iterator traitsIterator = item.iterator();
                while (traitsIterator.hasNext()) {
                    Trait trait = (Trait) traitsIterator.next();
                    String key = trait.getName();

                    //--- We only create percentage-based break points for
                    //---   numeric data
                    TraitDescriptor descriptor = descriptors.get(key);
                    int dataType = descriptor.getDataType();
                    boolean isNumeric = (dataType==TraitDescriptor.TYPE_FLOAT) ||
                                        (dataType==TraitDescriptor.TYPE_INTEGER);
                    if (!isNumeric) {
                        //--- Skip to the next trait
                        continue;
                    }

                    Float value = new Float(trait.getValue().value());
                    if (minValues.containsKey(key)) {
                        Float currentMin = (Float) minValues.get(key);
                        //--- i hate this primitive wrappers
                        //--- Float.compareTo() returns 0 if =,
                        //---   + if arg is smaller and - if arg bigger
                        if (currentMin.compareTo(value) > 0) {
                            minValues.put(key,value);
                        }
                    } else {
                        minValues.put(key,value);
                        maxValues.put(key,value);
                    }

                    Float currentMax = (Float) maxValues.get(key);
                    if (currentMax.compareTo(value) < 0) {
                        maxValues.put(key,value);
                    }
                }  //--- traitIterator.hasNext()
            }   //--- finding max and min
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }  //--- determineMinAndMax


    /**
     * OK, so now we have a collection (breakdowns) of DataBreakdownForTrait
     *  objects, but those objects don't have anything in them
     * Unlike strings and booleans, numeric traits (cost, speed, size, whatever)
     *  represent continuous data, meaning there's a max and a min and all the
     *  data falls somewhere in between the two, which is good because otherwise
     *  our nearest neighbor stuff doesn't work and this whole CBR/similarity
     *  engine is worthless
     * In this step we're going to create percentage-based break points
     *  with a granularity or 10 points. That means 11 break points - 0-100
     *  (0%, 10%, 20%, etc.). Obviously, the value at breakpoint 0 is whatever
     *  the min value was and the value at breakpoint 100 is the max value
     * So, for each numeric trait, we need to find min and max and then
     *  we need to determine the value at each breakpoint
     */
    protected void createPercentageBreakPoints(Items items) {
        try {
            determineMinAndMax(items);

            //--- Assumption: minValues and maxValues have the same keys
            Iterator traitsIterator = minValues.keySet().iterator();
            while (traitsIterator.hasNext()) {
                //--- Get the DataBreakdownForTrait object for this trait
                String traitName = (String) traitsIterator.next();
                DataBreakdownForNumericTrait breakdown =
                    (DataBreakdownForNumericTrait) breakdowns.get(traitName);

                float min = ((Float) minValues.get(traitName)).floatValue();
                float max = ((Float) maxValues.get(traitName)).floatValue();

                breakdown.createPercentageBreakPoints(min, max, stepIncrement);
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }  //--- putBreakPointsInBreakdowns


    public DataBreakdownForTrait getDataBreakdown(String traitName) {
        //--- Force a case so we can be consistent
        traitName = traitName.toLowerCase();

        DataBreakdownForTrait dataBreakdown =
            (DataBreakdownForTrait) breakdowns.get(traitName);

        return dataBreakdown;
    }


    public HashMap getDataBreakdowns() {
        return breakdowns;
    }


    /**
     * Each item has a series of traits (ID, cost, size, whatever)
     * We want to see what the different values are for each of these traits
     * So, for each trait, we're going to create a DataBreakdownForTrait object
     *  which will hold all the different values and frequency of each value
     *  for that trait
     * In this step, all we do is create a DataBreakdownForTrait for each
     *  trait the items have
     * Technically, we use the TraitDescriptors, which are meta-data about
     *  traits. They contain the trait's name and data type
     */
    protected void putTraitsInBreakdowns(Items items) {
        try {
            TraitDescriptors traitDescriptors = items.getTraitDescriptors();
            breakdowns = new HashMap();

            Iterator traitDescriptorIterator = traitDescriptors.iterator();
            while (traitDescriptorIterator.hasNext()) {
                //--- Get the name of the trait
                TraitDescriptor descriptor =
                    (TraitDescriptor) traitDescriptorIterator.next();
                String traitName = descriptor.getName();

                //--- Create a DataBreakdownForTrait object for this trait
                //--- Make sure to use the right DataBreakdownForTrait object.
                //---   There's a different one for each data type
                int dataType = descriptor.getDataType();
                DataBreakdownForTrait traitBreakdown = null;
                switch (dataType) {
                    case TraitDescriptor.TYPE_BOOLEAN:
                        traitBreakdown = new DataBreakdownForBooleanTrait(traitName);
                        break;
                    case TraitDescriptor.TYPE_STRING:
                        traitBreakdown = new DataBreakdownForStringTrait(traitName);
                        break;
                    default:
                        traitBreakdown = new DataBreakdownForNumericTrait(traitName);
                        break;
                }  //--- switch (dataType)

                //--- That's all. Lets store the object in our breakdowns collection
                traitName = traitName.toLowerCase();    //--- Force case for consistency
                breakdowns.put(traitName,traitBreakdown);
            }  //--- while (iterator.hasNext())
        }  //--- try
        catch (Exception e) {
            e.printStackTrace();
        }
    }  //--- putTraitsInBreakdowns


    protected void parseItems(Items items) {
        try {
            //--- We shouldn't need this, but just in case...
            //--- Clean out anything that was in our collection
            breakdowns = new HashMap();

            //--- Create one DataBreakdownForTrait for each trait
            putTraitsInBreakdowns(items);

            //--- For each DataBreakdownForTrait, grab max and min
            createPercentageBreakPoints(items);

            //--- For each item, add each of its properties to the proper bucket
            addDataToBreakPoints(items);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }  //--- parseItems
}  //--- DataBreakdownForTraits